/*
  ==============================================================================

    SonicCrypt Flux Gate
    Copyright (C) 2025 Sebastian Snkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#pragma once
#include <JuceHeader.h>
#include "PluginProcessor.h"

// --- ADVANCED VELOCITY VISUALIZER ---
class FluxVisualizer : public juce::Component {
public:
    std::vector<float> pattern;
    int currentStep = -1;

    // Callback liefert jetzt Index UND Level (0.0 - 1.0)
    std::function<void(int, float)> onStepLevelChanged;

    bool hitTest(int x, int y) override { return true; }

    // Hilfsfunktion: Mausposition in Step & Level umrechnen
    void handleMouseInput(const juce::MouseEvent& e) {
        int steps = (int)pattern.size();
        if (steps <= 0) return;

        auto bounds = getLocalBounds().toFloat();

        // 1. Welcher Step? (X-Achse)
        float widthPerStep = bounds.getWidth() / (float)steps;
        int stepIndex = (int)(e.position.x / widthPerStep);

        // 2. Welcher Level? (Y-Achse)
        // Y=0 ist oben (Level 1.0), Y=Height ist unten (Level 0.0)
        float level = 1.0f - (e.position.y / bounds.getHeight());

        // Clamp values
        if (stepIndex < 0) stepIndex = 0;
        if (stepIndex >= steps) stepIndex = steps - 1;
        level = juce::jlimit(0.0f, 1.0f, level);

        // Snap to Zero/Full
        if (level < 0.05f) level = 0.0f;
        if (level > 0.95f) level = 1.0f;

        if (onStepLevelChanged) onStepLevelChanged(stepIndex, level);
    }

    void mouseDown(const juce::MouseEvent& e) override {
        handleMouseInput(e);
    }

    void mouseDrag(const juce::MouseEvent& e) override {
        handleMouseInput(e);
    }

    void paint(juce::Graphics& g) override {
        // Hintergrund
        g.setColour(juce::Colour::fromString("FF111111"));
        g.fillRoundedRectangle(getLocalBounds().toFloat(), 4.0f);

        int steps = (int)pattern.size();
        if (steps <= 0) {
            g.setColour(juce::Colours::white);
            g.drawText("NO DATA", getLocalBounds(), juce::Justification::centred);
            return;
        }

        auto bounds = getLocalBounds().toFloat();
        float stepWidth = bounds.getWidth() / (float)steps;
        float totalHeight = bounds.getHeight();

        for (int i = 0; i < steps; ++i) {
            float val = pattern[i]; // 0.0 bis 1.0
            bool isCurrent = (i == currentStep);

            // X Position
            float x = i * stepWidth;
            float gap = 2.0f;

            // Hhe des Balkens
            float barHeight = totalHeight * val;

            // Rechteck von UNTEN nach OBEN
            juce::Rectangle<float> barRect(x + gap / 2,
                totalHeight - barHeight,
                stepWidth - gap,
                barHeight);

            // Farbe basierend auf Hhe
            if (val > 0.01f) {
                juce::Colour baseCol = juce::Colour::fromString("FF00BFFF");
                if (isCurrent) baseCol = juce::Colours::white;

                g.setColour(baseCol.withAlpha(0.5f + (val * 0.5f)));
                g.fillRoundedRectangle(barRect, 2.0f);
            }
            else {
                // Leerer Step (Ghost)
                g.setColour(juce::Colour::fromString("FF222222"));
                if (isCurrent) g.setColour(juce::Colours::grey.withAlpha(0.5f));

                juce::Rectangle<float> ghostRect(x + gap / 2, totalHeight - (totalHeight * 0.1f), stepWidth - gap, totalHeight * 0.1f);
                g.fillRoundedRectangle(ghostRect, 2.0f);
            }

            // Playhead Cursor Rahmen
            if (isCurrent) {
                g.setColour(juce::Colours::white.withAlpha(0.6f));
                juce::Rectangle<float> slotRect(x + gap / 2, 0, stepWidth - gap, totalHeight);
                g.drawRoundedRectangle(slotRect, 2.0f, 1.0f);
            }
        }
    }
};

// --- PRESET LIST ---
class PresetListComponent : public juce::ListBoxModel {
public:
    PresetListComponent(juce::ListBox& lb, std::function<void(juce::File)> onSelect) : listBox(lb), onSelection(onSelect) {}
    int getNumRows() override { return presets.size(); }
    void paintListBoxItem(int rowNumber, juce::Graphics& g, int width, int height, bool rowIsSelected) override {
        if (rowIsSelected) g.fillAll(juce::Colour::fromString("FF00BFFF").withAlpha(0.3f));
        g.setColour(rowIsSelected ? juce::Colour::fromString("FF00BFFF") : juce::Colours::white);
        g.setFont(14.0f);
        g.drawText(presets[rowNumber].getFileNameWithoutExtension(), 5, 0, width, height, juce::Justification::centredLeft);
    }
    void listBoxItemClicked(int row, const juce::MouseEvent&) override { if (onSelection) onSelection(presets[row]); }
    void refresh(const juce::File& folder) { presets.clear(); if (folder.isDirectory()) for (auto entry : juce::RangedDirectoryIterator(folder, false, "*.scp")) presets.add(entry.getFile()); listBox.updateContent(); }
private: juce::ListBox& listBox; juce::Array<juce::File> presets; std::function<void(juce::File)> onSelection;
};

// --- ABOUT ---
class AboutOverlay : public juce::Component {
public:
    AboutOverlay() {
        // OK Button
        okButton.setButtonText("OK");
        okButton.setColour(juce::TextButton::buttonColourId, juce::Colours::transparentBlack);
        okButton.onClick = [this] { setVisible(false); };
        addAndMakeVisible(okButton);

        // AGPL Lizenz Link
        licenseLink.setButtonText("GNU AGPL v3 License");
        licenseLink.setURL(juce::URL("https://www.gnu.org/licenses/agpl-3.0.html.en"));
        // Akzentfarbe fr Flux Gate (Deep Sky Blue)
        licenseLink.setColour(juce::HyperlinkButton::textColourId, juce::Colour::fromString("FF00BFFF"));
        addAndMakeVisible(licenseLink);

        // JUCE Link
        juceLink.setButtonText("Made with JUCE");
        juceLink.setURL(juce::URL("https://juce.com"));
        juceLink.setColour(juce::HyperlinkButton::textColourId, juce::Colours::white.withAlpha(0.6f));
        addAndMakeVisible(juceLink);
    }

    void paint(juce::Graphics& g) override {
        g.fillAll(juce::Colours::black.withAlpha(0.85f)); // Etwas dunklerer Background wie im Original Flux Gate Code

        // Layout (Einheitliche Gre wie bei den anderen Plugins)
        auto area = getLocalBounds().toFloat().withSizeKeepingCentre(500, 280);

        // Hintergrund Panel
        g.setColour(juce::Colour::fromString("FF1E1E1E").darker(0.2f));
        g.fillRoundedRectangle(area, 10.0f);
        g.setColour(juce::Colours::black);
        g.drawRoundedRectangle(area, 10.0f, 2.0f);

        auto content = area.reduced(20);

        // Icon (Flux Gate Blau)
        auto iconArea = content.removeFromLeft(80).removeFromTop(80);
        iconArea.setY(iconArea.getY() + 20);
        g.setColour(juce::Colour::fromString("FF00BFFF"));
        g.fillEllipse(iconArea);
        g.setColour(juce::Colours::black.withAlpha(0.3f));
        g.setFont(juce::FontOptions(50.0f).withStyle("Bold"));
        g.drawText("i", iconArea.toNearestInt(), juce::Justification::centred);

        content.removeFromLeft(20);

        // berschrift
        g.setColour(juce::Colours::white);
        g.setFont(juce::FontOptions(20.0f).withStyle("Bold"));
        g.drawText("About SonicCrypt Flux Gate", content.removeFromTop(30).toNearestInt(), juce::Justification::topLeft);

        // Entwickler
        g.setFont(juce::FontOptions(15.0f));
        g.drawText(juce::CharPointer_UTF8("Idea & Development: Sebastian S\xc3\xbc\x6ekler"), content.removeFromTop(25).toNearestInt(), juce::Justification::topLeft);

        // Beschreibung
        g.setColour(juce::Colours::grey);
        g.setFont(juce::FontOptions(13.0f));
        g.drawFittedText("Audio Triggered Euclidean Sequencer.\n\nVersion: 1.0", content.removeFromTop(50).toNearestInt(), juce::Justification::topLeft, 3);

        // Label "Licensed under:"
        g.drawText("Licensed under:", content.removeFromTop(20).toNearestInt(), juce::Justification::topLeft);

        // --- Steinberg Trademark Hinweis (Footer) ---
        auto footerArea = area.removeFromBottom(20).reduced(10, 0);
        g.setColour(juce::Colours::grey.withAlpha(0.5f));
        g.setFont(10.0f);
        g.drawFittedText("VST is a registered trademark of Steinberg Media Technologies GmbH.", footerArea.toNearestInt(), juce::Justification::centred, 1);
    }

    void resized() override {
        auto area = getLocalBounds().withSizeKeepingCentre(500, 280);

        // OK Button
        okButton.setBounds(area.getX() + 350, area.getBottom() - 45, 80, 25);

        // Lizenz Link
        licenseLink.setBounds(area.getX() + 120, area.getY() + 145, 150, 20);

        // JUCE Link 
        juceLink.setBounds(area.getX() + 20, area.getBottom() - 45, 120, 25);
    }

private:
    juce::TextButton okButton;
    juce::HyperlinkButton licenseLink;
    juce::HyperlinkButton juceLink;
};

// --- HEADER ---
class SonicCryptFluxGateAudioProcessorEditor : public juce::AudioProcessorEditor, public juce::Button::Listener, public juce::Timer
{
public:
    SonicCryptFluxGateAudioProcessorEditor(SonicCryptFluxGateAudioProcessor&);
    ~SonicCryptFluxGateAudioProcessorEditor() override;
    void paint(juce::Graphics&) override;
    void resized() override;
    void buttonClicked(juce::Button*) override;
    void timerCallback() override;

private:
    SonicCryptFluxGateAudioProcessor& audioProcessor;
    const int BASE_WIDTH = 1000; const int BASE_HEIGHT = 600;
    bool constructorFinished = false;

    juce::GroupComponent presetGroup{ "presets", "PRESETS" };
    juce::ListBox presetListBox; std::unique_ptr<PresetListComponent> presetListModel;
    juce::TextButton savePresetButton{ "SAVE" }, refreshButton{ "REFRESH" };
    juce::ImageButton logoButton; juce::TextButton copyrightBtn{ "info" };
    juce::Image logoImage;

    FluxVisualizer visualizer;
    juce::TextButton fluxButton{ "FLUX" };

    // 9 Regler fr 3x3 Grid
    juce::Slider stepsS, pulsesS, rotS;
    juce::Slider swingS, glitchS, rateS;
    juce::Slider smoothS, mixS, widthS;

    juce::Label stepsL, pulsesL, rotL;
    juce::Label swingL, glitchL, rateL;
    juce::Label smoothL, mixL, widthL;

    using SliderAtt = juce::AudioProcessorValueTreeState::SliderAttachment;
    std::unique_ptr<SliderAtt> stepsAtt, pulsesAtt, rotAtt;
    std::unique_ptr<SliderAtt> swingAtt, glitchAtt, rateAtt;
    std::unique_ptr<SliderAtt> smoothAtt, mixAtt, widthAtt;

    AboutOverlay aboutOverlay;
    std::shared_ptr<juce::FileChooser> fileChooser;
    void setupKnob(juce::Slider& s, juce::Label& l, std::unique_ptr<SliderAtt>& att, juce::String paramId, juce::String name);

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(SonicCryptFluxGateAudioProcessorEditor)
};